from typing import Optional, List

import pandas as pd
from matplotlib import pyplot as plt
from matplotlib.lines import Line2D

from config import FiguresConfig


class Utils:
    @staticmethod
    def aggregate(data: pd.DataFrame, aggregation: str) -> Optional[pd.DataFrame]:
        if aggregation == "mean":
            return data.mean()
        if aggregation == "median":
            return data.median()
        if aggregation == "min":
            return data.min()
        if aggregation == "max":
            return data.max()
        if aggregation == "std":
            return data.std()
        if aggregation == "var":
            return data.var()
        if aggregation == "mad":
            return data.mad()

        return None

    @staticmethod
    def _get_handles(ax: plt.Axes, special_legend: bool = False) -> (List[object], List[object]):
        if not special_legend:
            handles, labels = ax.get_legend_handles_labels()
            if type(handles[0]) is Line2D:
                for h in handles:
                    h._linewidth = FiguresConfig.LineWidth
        else:
            handles = ax.get_legend().legendHandles
            labels = [t._text for t in ax.get_legend().texts]

        return handles, labels

    @staticmethod
    def set_legend_from_vis(fig: plt.Figure, ax: plt.Axes, y_pos: float = 1.0, label_on_fig: bool = True, special_legend: bool = False):
        obj = fig if label_on_fig else ax

        if not special_legend:
            handles, labels = ax.get_legend_handles_labels()
            if type(handles[0]) is Line2D:
                for h in handles:
                    h._linewidth = FiguresConfig.LineWidth
        else:
            handles = ax.get_legend().legendHandles
            labels = [t._text for t in ax.get_legend().texts]

        obj.legend(
            handles,
            labels,
            loc="upper center",
            bbox_to_anchor=(0.5, y_pos),
            ncol=5,
            framealpha=0.95,
            edgecolor="#000000"
        )

    @staticmethod
    def set_legend_from_vis_with_ls(fig: plt.Figure, ax: plt.Axes, ls: List[str], y_pos: float = 1.0, label_on_fig: bool = True, special_legend: bool = False):
        obj = fig if label_on_fig else ax
        handles, labels = Utils._get_handles(ax, special_legend)

        for i, h in enumerate(handles):
            h.set_linestyle(ls[i])

        obj.legend(
            handles,
            labels,
            loc="upper center",
            bbox_to_anchor=(0.5, y_pos),
            ncol=5,
            framealpha=0.95,
            edgecolor="#000000"
        )

    @staticmethod
    def set_custom_legend(fig: plt.Figure, ax: plt.Axes, keys: List[str], colors: List[str], title: str = None, label_on_fig: bool = True, y_pos: float = 1.0):
        obj = fig if label_on_fig else ax

        lw = FiguresConfig.LineWidth
        if title is not None:
            custom_lines = [Line2D(xdata=(0, 1), ydata=(0, 0), color="#ffffff", linewidth=lw)] + \
                           [Line2D(xdata=(0, 1), ydata=(0, 0), color=c, linewidth=lw) for c in colors]
        else:
            custom_lines = [Line2D(xdata=(0, 1), ydata=(0, 0), color=c, linewidth=lw) for c in colors]

        leg = obj.legend(
            custom_lines,
            [title + ":"] + keys if title is not None else keys,
            loc="upper center",
            bbox_to_anchor=(0.5, y_pos),
            ncol=len(keys) + 1 if title is not None else len(keys),
            framealpha=0.95,
            edgecolor="#000000"
        )

        if title is not None:
            for vpack in leg._legend_handle_box.get_children()[:1]:
                for hpack in vpack.get_children():
                    hpack.get_children()[0].set_width(0)

    @staticmethod
    def add_costume_xlabel(fig: plt.Figure, label: str, y_pos: float = 0.0):
        fig.text(
            x=0.5,
            y=y_pos,
            s=label,
            horizontalalignment="center",
            verticalalignment="bottom",
            fontsize=FiguresConfig.RC_FONTS["axes"]["label"],
            fontweight="bold"
        )
